﻿using System;
using System.IO;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Diagnostics;

namespace DemoCSWF
{
    public partial class MainForm : Form
    {
        Timer timerREC = new Timer();
        Timer timerFPS = new Timer();

        public MainForm()
        {
            InitializeComponent();

            // Make the log file path the same as the process exe file path but different extension.
            string exePath = Process.GetCurrentProcess().MainModule.FileName;
            string logPath = System.IO.Path.ChangeExtension(exePath, ".txt");

            // Create a log file that is helpful for debugging.
            // It can be called before calling ScnLib_Initialize().
            ZDSoft.SDK.ScnLib_SetLogPathW(logPath, true);
        }

        private void MainForm_Load(object sender, EventArgs e)
        {
            // Initialize the SDK.
            // It must be called before calling most of the other APIs of the SDK.
            // Usually it's called only once at the start of the program.
            ZDSoft.SDK.ScnLib_InitializeW("");

             // Make the screen capture region frame visible. It's invisible by default.
            // If you don't like the style of the SDK built-in screen capture region frame, you may implement your own.
            ZDSoft.SDK.ScnLib_ShowCaptureRegionFrame(true);

            // We prefer 1080p webcam resolution. It's 320x240 by default.
            // If your webcam doesn't support the resolution you set, the SDK will choose the closest resolution automatically.
            ZDSoft.SDK.ScnLib_SetWebcamResolution(1920, 1080);

            // Prepare a timer for checking the recording status and updating recording time.
            timerREC.Tick += new EventHandler(On_Timer_REC);
            timerREC.Interval = (int)TimeSpan.FromSeconds(0.5).TotalMilliseconds;

            // Setup a timer to update the game FPS overlay for game capture mode
            timerFPS.Tick += new EventHandler(On_Timer_FPS);
            timerFPS.Interval = (int)TimeSpan.FromSeconds(1).TotalMilliseconds;
            timerFPS.Start();

            Update_UI();
        }

        private void MainForm_FormClosed(object sender, FormClosedEventArgs e)
        {
            // Uninitialize the SDK.
            // It must be called before exiting the process or there might be resource leak.
            // Usually it's called only once at the end of the program.
            ZDSoft.SDK.ScnLib_Uninitialize();
        }

        private void On_Timer_REC(object sender, EventArgs e)
        {
            // Since the SDK doesn't have any callback mechanism, 
            // you need to check the recording status periodically in a timerREC.
            // If it's recording then update the recording time info.
            if (ZDSoft.SDK.ScnLib_IsRecording())
            {
                StringBuilder t = new StringBuilder(11);

                // Get the current recording time string in the format of 'HH:MM:SS'.
                ZDSoft.SDK.ScnLib_GetRecTimeW(t);

                rec_time.Text = t.ToString();
            }
            // If it's not recording then stop recording on failure.
            // The recording may stop automatically if a fatal error occurs.
            // In this case you still need to do some cleanup and UI updates.
            else
            {
                Stop_Recording(false);

                Update_UI();
            }
        }

        private void On_Timer_FPS(object sender, EventArgs e)
        {
            // Is currently game capture mode and a Direct3D/OpenGL game window detected?
            if (ZDSoft.SDK.ScnLib_GetGameWnd() != IntPtr.Zero)
            {
                // Update the game FPS overlay.
                Update_Game_FPS_Overlay();
            }
        }

        // Create the game FPS DIB for drawing
        Bitmap imageFPS = new Bitmap(170, 70, System.Drawing.Imaging.PixelFormat.Format24bppRgb);

        private void Update_Game_FPS_Overlay()
        {
            // Prepare a graphics object to draw on the game FPS DIB.
            Graphics g = Graphics.FromImage(imageFPS);

            // Get the game FPS number.
            int fps = ZDSoft.SDK.ScnLib_GetGameFrameRate();

            // Place the game FPS number top vertically and right horizontally.
            StringFormat fpsFormat = new StringFormat();
            fpsFormat.Alignment = StringAlignment.Far;
            fpsFormat.LineAlignment = StringAlignment.Near;

            StringBuilder t = new StringBuilder(11);

            // Get the current recording time string in the format of 'HH:MM:SS'.
            ZDSoft.SDK.ScnLib_GetRecTimeW(t);

            // Place the REC time bottom vertically and right horizontally.
            StringFormat timeFormat = new StringFormat();
            timeFormat.Alignment = StringAlignment.Far;
            timeFormat.LineAlignment = StringAlignment.Far;

            // Clear the overlay background.
            g.FillRectangle(
                System.Drawing.Brushes.White,
                new System.Drawing.Rectangle(0, 0, imageFPS.Width, imageFPS.Height));

            // Display a camera icon on the left side.
            g.DrawIcon(System.Drawing.Icon.ExtractAssociatedIcon(
                System.Reflection.Assembly.GetEntryAssembly().ManifestModule.Name),
                3, 3);

            // Draw the game FPS number.
            g.DrawString(
                fps.ToString() + " fps",
                new Font("Impact", 30, System.Drawing.FontStyle.Regular, GraphicsUnit.Pixel),
                System.Drawing.Brushes.Gray,
                new System.Drawing.Rectangle(0, 0, imageFPS.Width, imageFPS.Height),
                fpsFormat);

            // Draw the REC time.
            g.DrawString(
                t.ToString(),
                new Font("Impact", 25, System.Drawing.FontStyle.Regular, GraphicsUnit.Pixel),
                ZDSoft.SDK.ScnLib_IsRecording() ? System.Drawing.Brushes.Red : System.Drawing.Brushes.LimeGreen,
                new System.Drawing.Rectangle(0, 0, imageFPS.Width, imageFPS.Height),
                timeFormat);

            // Lock the DIB and get the bits buffer pointer.
            System.Drawing.Imaging.BitmapData dibFPS = imageFPS.LockBits(
                new System.Drawing.Rectangle(0, 0, imageFPS.Width, imageFPS.Height),
                System.Drawing.Imaging.ImageLockMode.ReadOnly,
                System.Drawing.Imaging.PixelFormat.Format24bppRgb);

            // Show the game FPS overlay on the top-right corner of the game screen.
            ZDSoft.SDK.ScnLib_SetInGameOverlayPosition(ZDSoft.SDK.POSITION_TOP_RIGHT, 5, 5);
            ZDSoft.SDK.ScnLib_ShowInGameOverlay(dibFPS.Scan0, dibFPS.Width, dibFPS.Height, 24);

            // Don't forget to unlock the DIB.
            imageFPS.UnlockBits(dibFPS);
        }

        private void Update_UI()
        {
            // Update the radio boxes in the Screen Capture group.
            full_screen.Enabled = !ZDSoft.SDK.ScnLib_IsRecording();
            a_region_window.Enabled = !ZDSoft.SDK.ScnLib_IsRecording();
            pc_game_screen.Enabled = !ZDSoft.SDK.ScnLib_IsRecording();

            // Update the check boxes in the Audio Capture group.
            playback_audio.Checked = ZDSoft.SDK.ScnLib_IsRecordAudioSource(true); // true - the playback audio source (speakers / headphone)
            microphone_audio.Checked = ZDSoft.SDK.ScnLib_IsRecordAudioSource(false); // false - the recording audio source (microphone / line-in)

            // Update the check boxes in the Webcam Capture group
            // The webcam capture is enabled only when a webcam device is selected.
            // The selected webcam device index is a zero-based number. The first webcam device index is 0.
            if (ZDSoft.SDK.ScnLib_GetSelectedWebcamDevice() >= 0)
            {
                webcam_preview.Enabled = true;

                // If the webcam is being previewed you can get a non-zero preview window handle.
                if (ZDSoft.SDK.ScnLib_GetWebcamPreviewWnd() != IntPtr.Zero)
                {
                    webcam_preview.Checked = true;
                }
                // Otherwise the webcam is not being previewed.
                else
                {
                    webcam_preview.Checked = false;
                }

                // If it's set to record webcam only, screen capture will be discarded and you will get a full webcam video.
                if (ZDSoft.SDK.ScnLib_IsRecordWebcamOnly())
                {
                    full_webcam_video.Checked = true;
                    webcam_overlay.Checked = false;
                }
                // Otherwise you will get a screen recording video with a webcam overlay on it.
                else
                {
                    webcam_overlay.Checked = true;
                    full_webcam_video.Checked = false;
                }
            }
            // A negative index means no webcam device is selected.
            else
            {
                webcam_preview.Enabled = false;
                webcam_preview.Checked = false;
                webcam_overlay.Checked = false;
                full_webcam_video.Checked = false;
            }

            // Update the elements in the Output Video group.
            StringBuilder videoPath = new StringBuilder(260);

            // Get the output video file path to be created.
            ZDSoft.SDK.ScnLib_GetVideoPathW(videoPath, false);

            video_path.Text = videoPath.ToString();
            video_path.Enabled = !ZDSoft.SDK.ScnLib_IsRecording();
            change_video_path.Enabled = !ZDSoft.SDK.ScnLib_IsRecording();

            // Update the recording control buttons at the bottom.
            if (ZDSoft.SDK.ScnLib_IsRecording())
            {
                start_stop.Text = "Stop";
            }
            else
            {
                start_stop.Text = "Start";
            }

            if (ZDSoft.SDK.ScnLib_IsPaused())
            {
                pause_resume.Text = "Resume";
            }
            else
            {
                pause_resume.Text = "Pause";
            }

            pause_resume.Enabled = ZDSoft.SDK.ScnLib_IsRecording();
        }

        private void Start_Recording()
        {
            // If no foreground Direct3D/OpenGL game window is detected, prompt user what to do.
            if (ZDSoft.SDK.ScnLib_IsGameCaptureModeEnabled() && ZDSoft.SDK.ScnLib_GetGameWnd() == IntPtr.Zero)
            {
                string msgText = @"No foreground Direct3D/OpenGL game window is detected!

Please open a Direct3D/OpenGL game and place its window in the foreground, or the SDK will automatically fallback to standard screen capture mode.";

                MessageBox.Show(msgText, Text, MessageBoxButtons.OK, MessageBoxIcon.Information);
            }

            // Set the output video file path to be created.
            // The file path can include SDK-defined variables: <num>, <date> and <time>
            ZDSoft.SDK.ScnLib_SetVideoPathW(video_path.Text);

            // Start recording now.
            if (ZDSoft.SDK.ScnLib_StartRecording())
            {
                // Start the timer to check the recording status and update recording time.
                timerREC.Start();
            }
            else
            {
                // Do some cleanup if recording failed to start.
                Stop_Recording(false);
            }
        }

        private void Stop_Recording(bool bSuccessful)
        {
            // Stop the timer
            timerREC.Stop();

            // Stop the recording. It's OK to call it even if no recording is in progress.
            ZDSoft.SDK.ScnLib_StopRecording();

            StringBuilder filePath = new StringBuilder(260);

            if (bSuccessful)
            {
                // Get the saved video file path if the recording is done successfully.
                ZDSoft.SDK.ScnLib_GetVideoPathW(filePath, true);
            }
            else
            {
                // Get the saved log file path if the recording is failed.
                ZDSoft.SDK.ScnLib_GetLogPathW(filePath);
            }

            // Play the video file or show the log file.
            if (File.Exists(filePath.ToString()))
            {
                Process proc = new Process();
                proc.StartInfo.CreateNoWindow = false;
                proc.StartInfo.UseShellExecute = true;
                proc.StartInfo.FileName = filePath.ToString();
                proc.StartInfo.Verb = "open";
                proc.Start();
                proc.Close();
            }
        }

        private void start_stop_Click(object sender, EventArgs e)
        {
            if (!ZDSoft.SDK.ScnLib_IsRecording())
            {
                // Start recording if no recording is in progress.
                Start_Recording();
            }
            else
            {
                // Stop recording if a recording is in progress.
                Stop_Recording(true);
            }

            Update_UI();
        }

        private void pause_resume_Click(object sender, EventArgs e)
        {
            if (!ZDSoft.SDK.ScnLib_IsPaused())
            {
                // Pause recording if the recording is not paused.
                ZDSoft.SDK.ScnLib_PauseRecording();
            }
            else
            {
                // Resume recording if the recording is paused.
                ZDSoft.SDK.ScnLib_ResumeRecording();
            }

            Update_UI();
        }

        private void more_settings_Click(object sender, EventArgs e)
        {
            // Popup the SDK built-in settings dialog to let user configure more settings.
            // If you don't like the style of the SDK built-in settings dialog, you may implement your own.
            ZDSoft.SDK.ScnLib_ConfigureSettings(this.Handle);

            Update_UI();
        }

        private void change_video_path_Click(object sender, EventArgs e)
        {
            SaveFileDialog dlg = new SaveFileDialog();

            // Supported video file formats are MP4, FLV and AVI.
            dlg.Filter = "MP4 videos|*.mp4|FLV videos|*.flv|AVI videos|*.avi";
            dlg.DefaultExt = ".mp4";
            dlg.Title = "Set Output Video File Path";

            // Popup a save file dialog to let user change the output video file path.
            if (dlg.ShowDialog() == DialogResult.OK)
            {
                // Set the output video file path to be created.
                ZDSoft.SDK.ScnLib_SetVideoPathW(dlg.FileName);

                Update_UI();
            }

            dlg.Dispose();
        }

        private void browse_video_path_Click(object sender, EventArgs e)
        {
            StringBuilder videoPath = new StringBuilder(260);

            // Get the output video file path to be created.
            ZDSoft.SDK.ScnLib_GetVideoPathW(videoPath, false);

            // Remove the invalid file path chars that are used by SDK-defined variables: <num>, <date> and <time>
            videoPath = videoPath.Replace("<", "");
            videoPath = videoPath.Replace(">", "");

            // Get the folder where your recording videos are saved.
            string videoFolder = System.IO.Path.GetDirectoryName(videoPath.ToString());

            if (Directory.Exists(videoFolder))
            {
                // Browse the video folder in the Windows File Explorer.
                Process.Start("explorer.exe", videoFolder);
            }
        }

        private void full_screen_Checked(object sender, EventArgs e)
        {
            if (full_screen.Checked)
            {
                // Don't bind any capture window.
                ZDSoft.SDK.ScnLib_SetCaptureWnd(IntPtr.Zero, false);
                // Set all-zero coordinates to let SDK detect and use the full screen coordinates.
                ZDSoft.SDK.ScnLib_SetCaptureRegion(0, 0, 0, 0);
                // Make sure the game capture mode is disabled.
                ZDSoft.SDK.ScnLib_EnableGameCaptureMode(false);
            }
        }

        private void a_region_window_Checked(object sender, EventArgs e)
        {
            if (a_region_window.Checked)
            {
                int l = 0, t = 0, r = 0, b = 0;
                IntPtr hwnd = IntPtr.Zero;

                // Turn mouse cursor into a crosshair and let user select a region/window on screen.
                // To select a region:
                //   1) Move the mouse cursor to the top-left corner of the region you want to capture.
                //   2) Press and hold the left mouse button, and then move the mouse cursor to the bottom-right corner of the region you want to capture.
                //   3) Release the left mouse button, done. Or, you can click the right mouse button to cancel in the middle of the process.
                // To select a window:
                //   1) Move the mouse cursor over the window you want to capture and then the window will be highlighted.
                //   2) Click the left mouse button to select the pointing window.
                // To cancel selection, click the right mouse button.
                // If a region is selected, you will get the region coordinates and a zero window handle.
                // If a window is selected, you will get the window coordinates and a non-zero window handle.
                if (ZDSoft.SDK.ScnLib_SelectCaptureRegionW(ref l, ref t, ref r, ref b, ref hwnd, ""))
                {
                    // Bind the capture window if hwnd is non-zero.
                    // Or unbind the capture window if hwnd is zero.
                    ZDSoft.SDK.ScnLib_SetCaptureWnd(hwnd, true);
                    // Set the capture region coordinates.
                    ZDSoft.SDK.ScnLib_SetCaptureRegion(l, t, r, b);
                }

                // Make sure the game capture mode is disabled.
                ZDSoft.SDK.ScnLib_EnableGameCaptureMode(false);
            }
        }

        private void pc_game_screen_Checked(object sender, EventArgs e)
        {
            if (pc_game_screen.Checked)
            {
                // Don't bind any capture window. (You can bind a specific game window if you don't want SDK to detect foreground game window automatically)
                ZDSoft.SDK.ScnLib_SetCaptureWnd(IntPtr.Zero, false);
                // Set all-zero coordinates to let SDK detect and use the full screen coordinates if no game screen is detected.
                ZDSoft.SDK.ScnLib_SetCaptureRegion(0, 0, 0, 0);
            }

            // Enable/Disable the game capture mode.
            ZDSoft.SDK.ScnLib_EnableGameCaptureMode(pc_game_screen.Checked);
        }

        private void playback_audio_Checked(object sender, EventArgs e)
        {
            // Enable/Disable capture from the playback audio source (speakers / headphone).
            ZDSoft.SDK.ScnLib_RecordAudioSource(true, playback_audio.Checked);
        }

        private void microphone_audio_Checked(object sender, EventArgs e)
        {
            // Enable/Disable capture from the recording audio source (microphone / line-in).
            ZDSoft.SDK.ScnLib_RecordAudioSource(false, microphone_audio.Checked);
        }

        private void webcam_preview_Checked(object sender, EventArgs e)
        {
            // Open/Close the webcam preview window.
            ZDSoft.SDK.ScnLib_PreviewWebcam(webcam_preview.Checked, IntPtr.Zero, true, 0);
        }

        private void webcam_overlay_Checked(object sender, EventArgs e)
        {
            if (webcam_overlay.Checked)
            {
                // Enable webcam capture by selecting the first webcam device.
                ZDSoft.SDK.ScnLib_SelectWebcamDevice(0);
                // Don't record webcam only. Put the webcam overlay on the screen capture video.
                ZDSoft.SDK.ScnLib_RecordWebcamOnly(false);
            }
            else if (full_webcam_video.Checked == false)
            {
                // Close the webcam preview window.
                ZDSoft.SDK.ScnLib_PreviewWebcam(false, IntPtr.Zero, true, 0);
                // Disable webcam capture by selecting a negative index.
                ZDSoft.SDK.ScnLib_SelectWebcamDevice(-1);
            }

            Update_UI();
        }

        private void full_webcam_video_Checked(object sender, EventArgs e)
        {
            if (full_webcam_video.Checked)
            {
                // Enable webcam capture by selecting the first webcam device.
                ZDSoft.SDK.ScnLib_SelectWebcamDevice(0);
                // Record webcam only. Discard screen capture. Make a full webcam video.
                ZDSoft.SDK.ScnLib_RecordWebcamOnly(true);
            }
            else if (webcam_overlay.Checked == false)
            {
                // Close the webcam preview window.
                ZDSoft.SDK.ScnLib_PreviewWebcam(false, IntPtr.Zero, true, 0);
                // Disable webcam capture by selecting a negative index.
                ZDSoft.SDK.ScnLib_SelectWebcamDevice(-1);
            }

            Update_UI();
        }
    }
}
